'use strict';

/* --------------------------------------------------------------
 cart_dropdown.js 2018-06-13
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2018 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Enables the functionality of the cart dropdown, to open
 * via an event. This is needed to open the flyout after
 * an item is added to the cart
 */
gambio.widgets.module('cart_dropdown', ['xhr', gambio.source + '/libs/events'], function (data) {

    'use strict';

    // ########## VARIABLE INITIALIZATION ##########

    var $this = $(this),
        $window = $(window),
        $body = $('body'),
        $item = null,
        $target = null,
        isCartDropdownSticky = false,
        timer = null,
        animateTimer = null,
        ajax = null,
        ajaxData = {
        part1: 'header',
        part2: 'dropdown'
    },
        defaults = {
        // Default delay (in ms) after which the flyout closes
        delay: 5000,
        // Update request url
        url: 'shop.php?do=CartDropdown',
        // Selection of the container the result gets filled in
        fillTarget: 'header',
        // Duration that the count badge gets resized after adding an item to the basket
        countAnimation: 2000,
        // AJAX response content selectors
        selectorMapping: {
            cartDropdown: '.cart-dropdown',
            cartDropdownProducts: '.products',
            cartDropdownProductsCount: '.cart-products-count'
        }
    },
        options = $.extend(true, {}, defaults, data),
        module = {};

    // ########## EVENT HANDLER ##########

    /**
     * Helper function that scroll the list
     * down to the end
     * @private
     */
    var _scrollDown = function _scrollDown() {
        var $list = $this.find('.products-list'),
            height = $list.outerHeight() * 2; // Multiply with 2 to be sure that it gets scrolled to the bottom

        $list.animate({ 'scrollTop': height + 'px' }, 0);
    };

    /**
     * Triggers the mouseenter event
     * on the cartdropdown link
     * @param       {object}        e       jQuery event object
     * @private
     */
    var _open = function _open(e) {
        e.stopPropagation();

        if ($(defaults.selectorMapping.cartDropdownProductsCount).text() !== '0') {
            $(defaults.selectorMapping.cartDropdownProductsCount).removeClass('hidden');
        }

        $item.trigger('mouseenter', { prog: true });
    };

    /**
     * Triggers the mouseleave event
     * on the cartdropdown link
     * @param       {object}        e       jQuery event object
     * @private
     */
    var _close = function _close(e) {
        e.stopPropagation();
        $item.trigger('mouseleave', { prog: true });
    };

    /**
     * Helper function that resizes the count badge
     * after the add of an item to the basket for
     * a specific duration
     * @param       {string}    selector        Text value of the old badge (the count)
     * @param       {object}    config          The config for the badges from the ajax result content
     * @private
     */
    var _resizeCountBadge = function _resizeCountBadge(currentCount, config) {
        if (options.selectorMapping[config.selector] === undefined) {
            jse.core.debug.warn('The selector mapping "' + config.selector + '" doesn\'t exist.');
            return true;
        }

        var count = $(config.value).text(),
            $counts = $target.find(options.selectorMapping[config.selector]);

        if (currentCount !== count) {
            if (animateTimer) {
                clearTimeout(animateTimer);
            }

            $counts.addClass('big');
            animateTimer = setTimeout(function () {
                $counts.removeClass('big');
            }, options.countAnimation);
        }
    };

    /**
     * Updates the dropdown with data from
     * the server and opens the layer for a
     * certain time
     * @param       {object}        e               jQuery event object
     * @param       {boolean}       openDropdown    Defines if the dropdown shall be opened after update
     * @private
     */
    var _update = function _update(e, openDropdown) {
        if (ajax) {
            ajax.abort();
        }

        ajax = jse.libs.xhr.ajax({ url: options.url, data: ajaxData }).done(function (result) {
            if (options.selectorMapping[result.content.count.selector] === undefined) {
                jse.core.debug.warn('The selector mapping "' + result.content.count.selector + '" doesn\'t exist.');
                return true;
            }

            var count = $(options.selectorMapping[result.content.count.selector]).first().text();
            jse.libs.theme.helpers.fill(result.content, $target, options.selectorMapping);
            _resizeCountBadge(count, result.content.count);

            _scrollDown();

            if (openDropdown) {
                $this.trigger(jse.libs.theme.events.CART_OPEN(), []);
                timer = setTimeout(function () {
                    $this.trigger(jse.libs.theme.events.CART_CLOSE(), []);
                }, options.delay);
            }
        });
    };

    /**
     * Event handler that listens on the
     * mouseenter / leave events. If these
     * events are not triggered by this script
     * stop the timer, because the user has
     * moved the mouse cursor over the object
     * @param       {object}        e       jQuery event object
     * @param       {object}        d       JSON which contains the status if the program triggered the event
     * @private
     */
    var _preventExec = function _preventExec(e, d) {
        if ((!d || !d.prog) && timer) {
            clearTimeout(timer);
        }
    };

    /**
     * Sticky Cart Dropdown
     *
     * There are cases when the user adds something to the cart and this pops out but it cannot be seen cause
     * it is out of the viewport (e.g. user has scrolled to bottom). This method will make sure that the cart
     * dropdown is always visible by applying a "sticky" positioning to respective elements.
     *
     * @private
     */
    var _stickyCartDropdown = function _stickyCartDropdown() {
        // If the cart dropdown is not visible wait until the transition completes (see menu.js). 
        if (!$item.hasClass('open')) {
            var interval = setInterval(function () {
                if ($item.hasClass('open')) {
                    _stickyCartDropdown();
                    clearInterval(interval);
                }
            }, 100);

            isCartDropdownSticky = false;
            return;
        }

        var $cartDropdown = $(options.selectorMapping.cartDropdown);
        var cartDropdownOffset = $cartDropdown.offset();

        // Enable "sticky" position in order to make the cart dropdown visible to the user.
        if (!isCartDropdownSticky && cartDropdownOffset.top < $(window).scrollTop()) {
            $cartDropdown.css({
                position: 'fixed',
                top: 20,
                left: cartDropdownOffset.left
            });

            isCartDropdownSticky = true;
        }

        // Reset sticky position once the user has scrolled to top. 
        if (isCartDropdownSticky && cartDropdownOffset.top < $item.offset().top) {
            $cartDropdown.css({
                position: '',
                top: '',
                left: ''
            });

            isCartDropdownSticky = false;
        }
    };

    // ########## INITIALIZATION ##########

    /**
     * Init function of the widget
     *
     * @constructor
     */
    module.init = function (done) {

        $item = $this.find('> ul > li');
        $target = options.fillTarget ? $(options.fillTarget) : $this;

        $window.on('scroll', _stickyCartDropdown);

        $body.on(jse.libs.theme.events.CART_OPEN(), _open).on(jse.libs.theme.events.CART_CLOSE(), _close).on(jse.libs.theme.events.CART_UPDATE(), _update);

        $item.on('mouseenter mouseleave', _preventExec).on('mouseenter', _stickyCartDropdown);

        _scrollDown();

        if (location.search.search('open_cart_dropdown=1') !== -1) {
            $body.trigger(jse.libs.theme.events.CART_OPEN());
        }

        done();
    };

    // Return data to widget engine
    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
